/*
	contact.js

	Autor: sb

	Erstellt: 04.04.2024 

	Zweck: Hilfsroutinen für das Kontaktformular

	- Das Eingabefeld für die Nachricht (TEXTAREA) erweitert sich je nach Menge des eingegebenen Textes
	  automatisch nach unten hin.

	- Es werden Session-Informationen mitgeloggt wie das Betreten/Verlassen der Eingabefelder, Klicks auf
	  Eingabefelder und Buttons und das Absenden des Formulars, jeweils mit Zeitstempel (Millisekunden).
	  Dadurch soll erkennbar werden, ob mit dem Formular interagiert wird (menschlicher Benutzer) oder
	  nicht (maschineller Spambot).

	Voraussetzung: Das Eingabeformular wird durch das Perl-Script "contact.pl" verarbeitet.

	Anwendung:

	- In den <head>-Bereich der HTML-Datei (Template) folgendes einfügen:

	<script src="../contact.js" type="text/javascript"></script>

	- In das Eingabeformular der HTML-Datei (Template) folgendes verstecktes Eingabefeld einfügen:

	<input type="hidden" id="session" name="session" value="@SESSION@">
*/

"use strict";

function InitKontakt()
{
	var elems = document.getElementsByTagName('textarea');
	for (var i = 0; i < elems.length; i++)
	{
		var elem = elems[i];
		elem.setAttribute('style', 'height: ' + (elem.scrollHeight) + 'px; overflow-y: hidden;');
		elem.addEventListener('input', OnInputTextarea, false);
	}

	elems = document.getElementsByTagName('input');
	for (var i = 0; i < elems.length; i++)
	{
		var elem = elems[i];
		if (elem.type !== 'text' && elem.type !== 'password') continue;
		elem.setAttribute('onfocus', 'OnFocus(this);');
		elem.setAttribute('onblur', 'OnBlur(this);');
	}

	elems = document.getElementsByTagName('textarea');
	for (var i = 0; i < elems.length; i++)
	{
		var elem = elems[i];
		elem.setAttribute('onfocus', 'OnFocus(this);');
		elem.setAttribute('onblur', 'OnBlur(this);');
	}

	window.addEventListener('mousedown', OnClick);

	elems = document.getElementsByTagName('form');
	for (var i = 0; i < elems.length; i++)
	{
		var elem = elems[i];
		elem.setAttribute('onsubmit', 'OnSubmit(this);');
	}

	AddSessionData('Page', 'loaded');
}

function OnInputTextarea(e)
{
	this.style.height = 'auto';
	this.style.height = (this.scrollHeight) + 'px';
}

function OnFocus(elem)
{
	AddSessionData('Enter', GetElementText(elem));
}

function OnBlur(elem)
{
	AddSessionData('Leave', GetElementText(elem));
}

function OnClick(evt)
{
	var elem = evt.target;
	if (!elem.parentElement) return;
	AddSessionData('Click', GetElementText(elem));
}

function OnSubmit(elem)
{
	AddSessionData('Submit', GetElementText(elem));
}

function AddSessionData(action, data)
{
	var elemSession = document.getElementById('session');
	if (!elemSession) return;

	var sessionData = 
		'ms=' + encodeURIComponent(new Date().getTime()) +
		'&action=' + encodeURIComponent(action) +
		'&data=' + encodeURIComponent(data);

	console.log(decodeURIComponent(sessionData));

	if (elemSession.value)
		elemSession.value += '|';
	elemSession.value += sessionData;
}

function GetElementText(elem)
{
	var result = 'Tag=' + elem.tagName.toUpperCase();
	if (elem.id) result += '; ID=' + elem.id;
	if (elem.name) result += '; Name=' + elem.name;
	if (elem.value) result += '; Value=' + GetTextExcerpt(elem.value);
	var txt = GetTextExcerpt(elem.innerText || '');
	if (txt) result += '; Txt=' + txt;
	return result;
}

function GetTextExcerpt(txt)
{
	txt = txt.replace(/[\n]+/g, ' ');
	if (txt.length > 35) txt = txt.substring(0, 30) + '[...]';
	return txt;
}

document.addEventListener('DOMContentLoaded', InitKontakt, false);
